/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.rk.RockVideoPlayer.ui;

import android.rk.RockVideoPlayer.R;
import android.rk.RockVideoPlayer.RepeatingImageButton.RepeatListener;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.os.Handler;
import android.os.Message;
import android.os.SystemClock;
import android.util.AttributeSet;
import android.util.Log;
import android.widget.ImageView;
import android.view.View.OnClickListener;
import android.view.View.OnLongClickListener;
import android.view.KeyEvent;
import android.view.View;
import android.view.MotionEvent;
import android.content.res.Resources;

/**
 * Button with click-animation effect.
 */
public class ColorButton extends ImageView implements OnClickListener, OnLongClickListener {
    private static final String TAG = "ColorButton";
    static final int CLICK_FEEDBACK_INTERVAL = 20;
    static final int CLICK_FEEDBACK_DURATION = 850;
    static final int REPEAT_DELAY = 250;
    static final int MESSAGE_REPEAT = 0;

    private long mInterval = 460;
    private long mStartTime;
    private int mRepeatCount;
    Drawable mButtonBackground;
    Drawable mButton;
    float mTextX;
    float mTextY;
    long mAnimStart;
    OnClickListener mListener;
    OnLongClickListener mLongListener;
    RepeatListener mRepListener;
    boolean ispress = false;
    boolean isavailability = false;
    boolean isLongClick = false;
    boolean isInRect = true;
    boolean isFinishAnimation = false;
    private Handler mHandler = null;
    private Handler mHandlerRepeat = new Handler();
    private Runnable mRunnable = null;

    public ColorButton(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
        setOnClickListener(this);
        setOnLongClickListener(this);
    }

    public void setOnClick(OnClickListener listener) {
        mListener = listener;
    }

    public void setOnLongClick(OnLongClickListener listener) {
        mLongListener = listener;
    }

    public void setRunnable(Runnable runnable) {
        mHandler = new Handler() {
            @Override
            public void dispatchMessage(Message msg) {
                switch (msg.what) {
                    case MESSAGE_REPEAT:
                        if (isavailability && isInRect) {
                            mHandlerRepeat.post(mRunnable);
                            mHandler.sendEmptyMessageDelayed(MESSAGE_REPEAT, REPEAT_DELAY);
                        }
                        break;
                }
                super.dispatchMessage(msg);
            }
        };
        mRunnable = runnable;
    }

    public void onClick(View view) {
        if (!isLongClick && mListener != null && isInRect) {
            mListener.onClick(this);
        }
    }

    public boolean onLongClick(View v) {
        if (mLongListener != null && isInRect) {
            animateClickFeedback();
            mLongListener.onLongClick(v);
            isavailability = false;
            ispress = false;
            isLongClick = true;
        }
        return false;
    }

    private void init() {
        setBackgroundDrawable(null);
        Resources res = getResources();
        mButtonBackground = res.getDrawable(R.drawable.button_bk_select);
        mAnimStart = -1;
    }

    public void setButtonDrawable(int id) {
        mButton = getResources().getDrawable(id);
    }


    private void drawMagicFlame(int duration, Canvas canvas) {
        int alpha = 255 - 255 * duration / CLICK_FEEDBACK_DURATION;
        mButtonBackground.setBounds(0, 0, getWidth(), getHeight());
        mButtonBackground.setAlpha(alpha);
        mButtonBackground.draw(canvas);
//        float mscale = duration / 365.0f;
//        Log.d(TAG, "________   duration = " + duration + "   mscale = " + mscale);
    }

    @Override
    public void onDraw(Canvas canvas) {
//    	Log.d(TAG, " _________ mAnimStart = " + mAnimStart + "    isPressed() = " + isPressed());
        if (mAnimStart != -1) {
            int animDuration = (int) (System.currentTimeMillis() - mAnimStart);

            if (animDuration >= CLICK_FEEDBACK_DURATION) {
                mButtonBackground.clearColorFilter();
                mButtonBackground.setAlpha(255);
                mAnimStart = -1;
                isFinishAnimation = true;
            } else {
                drawMagicFlame(animDuration, canvas);
                postInvalidateDelayed(CLICK_FEEDBACK_INTERVAL);
            }
        } else if (ispress) {
            drawMagicFlame(0, canvas);
        }
        super.onDraw(canvas);
    }

    public void animateClickFeedback() {
        mAnimStart = System.currentTimeMillis();
        invalidate();
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        int a = event.getAction();
//        Log.d(TAG, " _________   onTouchEvent(),  a = " + a + "   (" + event.getX() + ", " + event.getY() + ")" + "    (" + this.getWidth() + ", " + this.getHeight() + ")");
        removerButtonHandler.removeMessages(0);
        removerButtonHandler.sendEmptyMessageDelayed(0, REPEAT_DELAY * 2);
        if (a == MotionEvent.ACTION_CANCEL) {
            invalidate();
        } else if (a == MotionEvent.ACTION_DOWN) {
            isLongClick = false;
            ispress = true;
            mAnimStart = -1;
            isavailability = true;
            isInRect = true;
            isFinishAnimation = false;
            invalidate();
            if (mRunnable != null) {
                mHandler.removeMessages(MESSAGE_REPEAT);
                mHandler.sendEmptyMessage(MESSAGE_REPEAT);
            }
            if (mRepListener != null)
                mRepListener.doBeginLongclick();
        } else if (a == MotionEvent.ACTION_UP && isavailability && isInRect) {
            ispress = false;
            isavailability = false;
            if (!isLongClick)
                animateClickFeedback();

            if (mRepListener != null) {
                mRepListener.doAfterLongclick();
                removeCallbacks(mRepeater);
                if (mStartTime != 0) {
                    doRepeat(true);
                    mStartTime = 0;
                }
            }
        } else if (a == MotionEvent.ACTION_MOVE && isavailability && isInRect) {
            ispress = true;
        }
        if ((event.getX() < 0 || event.getY() < 0
                || event.getX() > this.getWidth() || event.getY() > this.getHeight())
                && isInRect && isavailability) {
            ispress = false;
            isavailability = false;
            isInRect = false;
            animateClickFeedback();
        }
        return super.onTouchEvent(event);
    }

    private Handler removerButtonHandler = new Handler() {
        @Override
        public void handleMessage(Message msg) {
            if ((boolean) (mAnimStart == -1) && !isFinishAnimation) {
                ispress = false;
                isavailability = false;
                animateClickFeedback();
            }
        }
    };


    private Runnable mRepeater = new Runnable() {
        public void run() {
            doRepeat(false);
            if (isPressed()) {
                postDelayed(this, mInterval);
                //post(this);
            }
        }
    };

    @Override
    public boolean onKeyUp(int keyCode, KeyEvent event) {
        switch (keyCode) {
            case KeyEvent.KEYCODE_DPAD_CENTER:
            case KeyEvent.KEYCODE_ENTER:
                // remove the repeater, but call the hook one more time
                removeCallbacks(mRepeater);
                if (mStartTime != 0) {
                    doRepeat(true);
                    mStartTime = 0;
                }
        }
        return super.onKeyUp(keyCode, event);
    }

    @Override
    public boolean performLongClick() {
        if (mRepListener != null) {
            mStartTime = SystemClock.elapsedRealtime();
            mRepeatCount = 0;
            post(mRepeater);
        }
        return true;
    }

    private void doRepeat(boolean last) {
        long now = SystemClock.elapsedRealtime();
        if (mRepListener != null) {
            mRepListener.onRepeat(this, now - mStartTime, last ? -1 : mRepeatCount++);
            mRepListener.isClick(last);
        }
    }

    public void setRepeatListener(RepeatListener l, long interval) {
        mRepListener = l;
        mInterval = interval;
    }

    public interface RepeatListener {
        void onRepeat(View v, long duration, int repeatcount);

        void isClick(boolean click);

        void doAfterLongclick();

        void doBeginLongclick();
    }
}
